/*
 * Copyright (c) 1999-2001 Lutris Technologies, Inc. All Rights
 * Reserved.
 * 
 * This source code file is distributed by Lutris Technologies, Inc. for
 * use only by licensed users of product(s) that include this source
 * file. Use of this source file or the software that uses it is covered
 * by the terms and conditions of the Lutris Enhydra Development License
 * Agreement included with this product.
 * 
 * This Software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF
 * ANY KIND, either express or implied. See the License for the specific terms
 * governing rights and limitations under the License.
 * 
 * Contributor(s):
 * 
 * $Id: StartTag.java,v 1.1.1.1 2001/01/05 05:11:17 sese0235 Exp $
 */

package de.kxml.parser;

import java.io.IOException;
import java.util.*;
import de.kxml.*;
import de.kxml.io.*;

/**
 * A class for events indicating the start of a new element
 */
public class StartTag extends ParseEvent {
    String    name;
    String    namespace;
    Vector    attributes;
    boolean   degenerated;
    StartTag  previous;
    PrefixMap prefixMap;

    /**
     * creates a new StartTag. The attributes are not copied and may
     * be reused in e.g. the DOM. So DO NOT CHANGE the attribute
     * vector after handing over, the effects are undefined
     */
    public StartTag(StartTag previous, String namespace, String name, 
		    Vector attributes, boolean degenerated) {
	this.previous = previous;
	this.namespace = namespace == null ? "" : namespace;
	this.name = name;
	this.attributes = attributes;
	this.degenerated = degenerated;
	prefixMap = previous == null ? new PrefixMap() : previous.prefixMap;
    }

    /**
     * creates a new StartTag from the given string (including
     * attributes) without brakets. It is assumed that the opening
     * braket is already consumed.
     */
    public StartTag(StartTag previous, 
		    LookAheadReader reader) throws IOException {
	this.previous = previous;
	prefixMap = previous == null ? new PrefixMap() : previous.prefixMap;
	name = reader.readTo("/> \t\n\r");

	// System.out.println ("name: ("+name+")");
	attributes = new Vector();

	while (true) {
	    reader.skipWhitespace();

	    int c = reader.peek();

	    if (c == '/') {
		degenerated = true;

		reader.read();
		reader.skipWhitespace();

		if (reader.read() != '>') {
		    throw new ParseException("illegal element termination", 
					     reader);
		} 

		break;
	    } 

	    if (c == '>') {
		reader.read();

		break;
	    } 

	    if (c == -1) {
		throw new ParseException("unexpected eof", reader);
	    } 

	    String attrName = reader.readTo('=');

	    while (attrName.length() > 0 
		   && attrName.charAt(attrName.length() - 1) <= ' ') {
		attrName = attrName.substring(0, attrName.length() - 1);
	    }

	    reader.read();
	    reader.skipWhitespace();

	    int delimiter = reader.read();

	    if (delimiter != '\'' && delimiter != '"') {
		throw new ParseException("<" + name 
					 + ">: invalid delimiter: " 
					 + (char) delimiter, reader);
	    } 

	    String attrValue = Xml.decode(reader.readTo((char) delimiter));

	    attributes.addElement(new Attribute(null, attrName, attrValue));
	    reader.read();    // skip endquote
	} 
    }

    /**
     * resolves namespaces from given prefixMap and
     * updates prefixMap with new namespace definitions
     */
    public void fixNamespaces() {
	boolean any = false;

	for (int i = attributes.size() - 1; i >= 0; i--) {
	    Attribute attr = (Attribute) attributes.elementAt(i);
	    String    name = attr.getName();
	    int       cut = attr.getName().indexOf(':');
	    String    prefix;

	    if (cut != -1) {
		prefix = name.substring(0, cut);
		name = name.substring(cut + 1);
	    } else if (name.equals("xmlns")) {
		prefix = name;
		name = "";
	    } else {
		continue;
	    }

	    if (!prefix.equals("xmlns")) {
		any = true;
	    } else {
		prefixMap = new PrefixMap(prefixMap, name, attr.getValue());

		// System.out.println (prefixMap);
		attributes.removeElementAt(i);
	    } 
	} 

	int len = attributes.size();

	if (any) {
	    for (int i = 0; i < len; i++) {
		Attribute attr = (Attribute) attributes.elementAt(i);
		String    attrName = attr.getName();
		int       cut = attr.getName().indexOf(':');

		if (cut != -1) {
		    String attrPrefix = attrName.substring(0, cut);

		    attrName = attrName.substring(cut + 1);

		    String attrNs = prefixMap.getNamespace(attrPrefix);

		    if (attrNs != null) {
			attributes.setElementAt(new Attribute(attrNs, attrName, attr.getValue()), 
						i);
		    } 
		} 
	    } 
	} 

	int    cut = name.indexOf(':');
	String prefix;

	if (cut == -1) {
	    prefix = "";
	} else {
	    prefix = name.substring(0, cut);
	    name = name.substring(cut + 1);
	} 

	namespace = prefixMap.getNamespace(prefix);

	if (namespace == null && prefix != "") {
	    throw new RuntimeException("undefined prefix: " + prefix + " in " 
				       + prefixMap + " at " + this);
	} 
    } 

    /**
     * returns the attribute at the given index position
     */
    public Attribute getAttribute(int index) {
	return (Attribute) attributes.elementAt(index);
    } 

    /**
     * returns the local attribute with the given name.  convenience
     * method for getAttribute (null, name);
     */
    public Attribute getAttribute(String name) {
	return getAttribute("", name);
    } 

    /**
     * returns the local attribute with the given qualified name.
     */
    public Attribute getAttribute(String namespace, String name) {
	int len = attributes.size();

	if (namespace == null) {
	    namespace = "";
	} 

	for (int i = 0; i < len; i++) {
	    Attribute attr = (Attribute) attributes.elementAt(i);

	    if (namespace.equals(attr.getNamespace()) 
		    && attr.getName().equals(name)) {
		return attr;
	    } 
	} 

	return null;
    } 

    /**
     * returns the number of attributes
     */
    public int getAttributeCount() {
	return attributes.size();
    } 

    /**
     * returns the attribute vector
     */
    public Vector getAttributes() {
	return attributes;
    } 

    /**
     * retrurns if the element is degenerated
     */
    public boolean getDegenerated() {
	return degenerated;
    } 

    /**
     * returns the (local) name of the element started
     */
    public String getName() {
	return name;
    } 

    /**
     * returns namespace
     */
    public String getNamespace() {
	return namespace;
    } 

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @see
     */
    public PrefixMap getPrefixMap() {
	return prefixMap;
    } 

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @see
     */
    public StartTag getPrevious() {
	return previous;
    } 

    /**
     * returns the integer constant assigned to this event type
     * (Xml.START_TAG).
     */
    public int getType() {
	return Xml.START_TAG;
    } 

    /**
     * returns the value of the attribute with the given name,
     * or null, if not existiong
     */
    public String getValue(String attrname) {
	Attribute attr = getAttribute(attrname);

	return attr == null ? null : attr.getValue();
    } 

    /**
     * Method declaration
     *
     *
     * @param start
     *
     * @return
     *
     * @see
     */
    public boolean endCheck(StartTag start) {
	throw new RuntimeException("unexpected element: " + this);
    } 

    /**
     * Simplified (!) toString method for debugging
     * purposes only
     */
    public String toString() {
	StringBuffer buf = new StringBuffer("<");

	buf.append(name);

	for (int i = 0; i < attributes.size(); i++) {
	    Attribute attr = getAttribute(i);

	    buf.append(" ");
	    buf.append(attr.getName());
	    buf.append("=\"");
	    buf.append(attr.getValue());
	    buf.append("\"");
	} 

	buf.append(">");

	if (lineNumber != -1) {
	    buf.append("<!-- #" + lineNumber + " -->");
	} 

	return buf.toString();
    } 

    /**
     * Method declaration
     *
     *
     * @param map
     *
     * @see
     */
    public void setPrefixMap(PrefixMap map) {
	this.prefixMap = map;
    } 

}

